/***************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Id: rcXPacket.cpp 7537 2016-04-11 13:22:35Z Robert $
   Last Modification:
    $Author: Robert $
    $Date: 2016-04-11 15:22:35 +0200 (Mo, 11 Apr 2016) $
    $Revision: 7537 $

   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no

   Description:
    Implementation of the rcX packet class

   Changes:

     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     10        11.04.2016  RM       Added misssing information displayed by the netHost application
                                    - ulGetPacketCnt / ulPutPacketCnt
                                    - ulHSKFlagSize / ulNumOfBlocks / ulIOInAreaCnt / ulIOOutAreaCnt
                                    - ulMBXSize of the system channel
      9        13.10.2015  RM       Writing ulCommunicationError to CHANNEL_INFO struture
      8        21.02.2012  SS       - Timeout was passed mistakenly as time-to-reset value
                                      in reset command (now set to 500ms)
      7        07.10.2011  SS       Reduce the timeout of RPC involved in connection establishment
      6        18.04.2011  SS       - Remove parameter checks from cifX API calls, as this
                                      is already done by cifX API calling layer
      5        01.04.2011  SS       - Connection state (m_fConnectionValid)
                                      moved to base class
                                    - Bugfix: Device name cleared at deinit
      4        27.07.2010  SS       - Bugfix: Read system information block causes
                                      out-of-bounds buffer access
                                    - Unused variables removed
      3        25.05.2010  SS       - Validate command id of packet reply to reject
                                      invalid packets from device side

      2        27.04.2010  SS       - Bugfix: Init() doesn't refresh device information
                                      after reconnect

      1        09.04.2010  SS       - xChannelBusState(): Implementation added
                                    - xChannelControlBlock(): Implementation added
                                    - GenerateChannelInfo(): Read Status and COS Flags
                                    - Bugfix: xChannelCommonStatusBlock() Read status block
                                      via system channel, because access via communication
                                      channel returns status block from wrong channel
                                    - Bugfix: xSysdeviceFindFirst/NextFile() Wrong len of
                                      packet data supplied
                                    - Bugfix: ReadChannelInfo() Function returns no error
                                      on wrong channel type

**************************************************************************************/


/////////////////////////////////////////////////////////////////////////////
/// \file rcXPacket.cpp
/// Implementation of the rcXPacket class
/////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "rcXpacket.h"
#include "TransportLayer.h"
#include "DeviceHandler.h"
#include "PhysicalInterface.h"
#include "HelperFunctions.h"

/////////////////////////////////////////////////////////////////////////////
/// Constructor
/////////////////////////////////////////////////////////////////////////////
CrcXPacket::CrcXPacket( CTransportLayer*  pcTransportLayer,
                        CEndpoint*        pcEndpoint)
: CDataLayer(pcTransportLayer)
, m_pcEndpoint(pcEndpoint)
, m_ulSendPktRefCnt(0)
, m_lRefCount(0)
, m_hReConnectEvent(NULL)
{
  InitializeCriticalSection(&m_tcsChannels);

  memset( &m_tDeviceInfo,   0,  sizeof(m_tDeviceInfo));
  memset( &m_tSystemInfo,   0,  sizeof(m_tSystemInfo));

  /*if(NULL != pcTransport)
  {
    // Set Event or callback
    m_hReConnectEvent    = ::CreateEvent(NULL, FALSE, FALSE, NULL);
    if(NULL != m_hReConnectEvent)
    {
      bool fRet = pcTransport->RegisterReconnectEvent( this, HIL_TRANSPORT_TYPE_RCX_PACKET, m_hReConnectEvent);
      if(false == fRet)
      {
        CloseHandle(m_hReConnectEvent);
        m_hReConnectEvent = NULL;
      }
    }
  }
  */
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CrcXPacket::~CrcXPacket(void)
{
  //// Check Reconnect event was set
  //if(NULL != m_hReConnectEvent)
  //{
  //  m_pcTransport->UnRegisterReconnectEvent(this);
  //  CloseHandle(m_hReConnectEvent);
  //  m_hReConnectEvent = NULL;
  //}

  Deinit();

  DeleteCriticalSection(&m_tcsChannels);
}

/////////////////////////////////////////////////////////////////////////////
/// Initialisation function
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::Init(void)
{
  int32_t          lRet;

  /* Use device number zero, cause only one device supported on interface */
  m_szDeviceName.append( NX_DEFAULT_NAME);
  m_szDeviceName.append( "0");

  if(CIFX_NO_ERROR == (lRet = ReadHardwareIdentify( &m_tDeviceInfo)))
  {
    uint8_t bInternalChannelIdx = RCXPACKET_SYSTEMCHANNEL_INTERNAL;
    for( uint32_t ulAreaIndex = 0; (ulAreaIndex < NETX_MAX_SUPPORTED_CHANNELS); ulAreaIndex++)
    {
      CHANNEL_INFO_T tChannelInfo = {0};
      if(CIFX_NO_ERROR == ReadChannelInfo(ulAreaIndex, &tChannelInfo))
      {
        EnterCriticalSection(&m_tcsChannels);
        m_cmChannels[bInternalChannelIdx++] = tChannelInfo;
        LeaveCriticalSection(&m_tcsChannels);
      }
    }

    /* Read additional informations. This command may not supported on older bootloader versions */
    ReadSysInfoBlock(&m_tSystemInfo);

    m_fConnectionValid = true;

    /* Device supports packets, so insert it into available data layers */
    m_pcEndpoint->m_cmDataLayers[HIL_TRANSPORT_TYPE_RCX_PACKET] = this;
    m_pcEndpoint->m_pcDefaultDataLayer                          = this;

    BOARD_INFORMATION tBoardInfo = {0};

    GenerateBoardInformation(&tBoardInfo);

    CDevice* pcDevice = m_pcEndpoint->AddDevice(tBoardInfo);

    EnterCriticalSection(&m_tcsChannels);
    CHANNEL_MAP::iterator iter = m_cmChannels.begin();

    while(iter != m_cmChannels.end())
    {
      CHANNEL_INFORMATION tChannelInfo = {0};

      GenerateChannelInfo(iter->second, &tChannelInfo);

      pcDevice->AddChannel(iter->second.ulChannelId, &tChannelInfo);

      iter++;
    }

    LeaveCriticalSection(&m_tcsChannels);
    TRACE("%s: Datalayer initialized - rcX packet transfer",
          m_pcEndpoint->m_pcInterface->GetInterfaceName().c_str());
  } else
  {
    Deinit();

    TRACE("%s: Datalayer not supported - rcX packet transfer",
          m_pcEndpoint->m_pcInterface->GetInterfaceName().c_str());
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Deinitialisation function
/////////////////////////////////////////////////////////////////////////////
void CrcXPacket::Deinit(void)
{
  m_fConnectionValid = false;

  EnterCriticalSection(&m_tcsChannels);
  m_cmChannels.clear();
  LeaveCriticalSection(&m_tcsChannels);
  m_szDeviceName.clear();
  ZeroMemory(&m_tDeviceInfo, sizeof(m_tDeviceInfo));
}

/////////////////////////////////////////////////////////////////////////////
/// Receive data function
///   \param    ptPacket  Pointer to a transport packet buffer
///   \return   true      if packet should be deleted by caller
/////////////////////////////////////////////////////////////////////////////
bool CrcXPacket::ReceiveData(PHIL_TRANSPORT_PACKET ptPacket)
{
  if(NULL == ptPacket)
    return true;

  bool    fRet = true;

  // Get channel number and place the packet into the corresponding queue
  BYTE bChannel = ptPacket->tHeader.bChannel;

  if(ptPacket->tHeader.ulLength > 0)
  {
    EnterCriticalSection(&m_tcsChannels);

    CHANNEL_MAP::iterator iter = m_cmChannels.find(bChannel);

    if(iter != m_cmChannels.end())
    {
      CHANNEL_INFO_T& tChannelInfo = iter->second;

      tChannelInfo.cvRecvQueue.push_back(ptPacket);
      tChannelInfo.ulRecvCount++;

      /* Signal new packet received */
      ReleaseSemaphore(tChannelInfo.hRecvSemaphore, 1, NULL);
      fRet = false;
    }

    LeaveCriticalSection(&m_tcsChannels);
  }

  return fRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Send complete indication
///   \param  ptHeader Pointer to a transport packet header
///   \return void
/////////////////////////////////////////////////////////////////////////////
void CrcXPacket::SendComplete(PHIL_TRANSPORT_HEADER /*ptHeader*/)
{
}

/////////////////////////////////////////////////////////////////////////////
/// Enumerate devices
///   \param    ulBoard       Number of Board
///   \param    ulSize        Size of buffer
///   \param    pvBoardInfo   Pointer of buffer (BOARD_INFORMATION)
///   \return   CIFX_NO_ERROR, if succes
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::EnumDevice(uint32_t ulBoard,
                            uint32_t ulSize,
                            void*         pvBoardInfo)
{
  int32_t lRet = CIFX_NO_ERROR;

  if(NULL == pvBoardInfo)
  {
    lRet = CIFX_INVALID_POINTER;
  } else if(sizeof(BOARD_INFORMATION) > ulSize)
  {
    lRet = CIFX_INVALID_BUFFERSIZE;
  } else if(0 != ulBoard)
  {
    /* Now CrcXPacket supports only one board, so check number of board */
    lRet = CIFX_INVALID_BOARD;
  } else
  {
    BOARD_INFORMATION*    ptBoardInfo   = (BOARD_INFORMATION*)pvBoardInfo;

    GenerateBoardInformation(ptBoardInfo);
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Enumerate devices
///   \param    ulBoard         Number of Board
///   \param    ulChannel       Number of Channel
///   \param    ulSize          Size of buffer
///   \param    pvChannelInfo   Pointer of buffer (CHANNEL_INFORMATION)
///   \return   CIFX_NO_ERROR, on succes
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::EnumChannel(uint32_t  ulBoard,
                             uint32_t  ulChannel,
                             uint32_t  ulSize,
                             void*          pvChannelInfo)
{
  int32_t lRet = CIFX_INVALID_CHANNEL;

  if(NULL == pvChannelInfo)
  {
    lRet = CIFX_INVALID_POINTER;
  } else if(sizeof(CHANNEL_INFORMATION) > ulSize)
  {
    lRet = CIFX_INVALID_BUFFERSIZE;
  } else if(0 != ulBoard)
  {
    /* Now CrcXPacket supports only one board, so check number of board */
    lRet = CIFX_INVALID_BOARD;
  } else if(ulChannel > MAXBYTE)
  {
    lRet = CIFX_INVALID_CHANNEL;

  } else
  {
    EnterCriticalSection(&m_tcsChannels);

    CHANNEL_MAP::iterator iter = m_cmChannels.begin();
    while (iter != m_cmChannels.end())
    {
      if (iter->second.ulChannelId == ulChannel)
        break;
      iter++;
    }

    if(iter != m_cmChannels.end())
    {
      CHANNEL_INFO_T& tChannelInfo = iter->second;

      CHANNEL_INFORMATION*    ptChannelInfo = (CHANNEL_INFORMATION*)pvChannelInfo;

      GenerateChannelInfo(tChannelInfo, ptChannelInfo);

      lRet = CIFX_NO_ERROR;
    }

    LeaveCriticalSection(&m_tcsChannels);
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Restart device
///   \param    szBoard       Name of Board
///   \return   CIFX_NO_ERROR on succes
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::RestartDevice(char* szBoard)
{
  UNREFERENCED_PARAMETER(szBoard);
  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/////////////////////////////////////////////////////////////////////////////
/// Read device information
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadHardwareIdentify( RCX_HW_IDENTIFY_CNF_DATA_T* ptHardwareIdentify)
{
  if(NULL == ptHardwareIdentify)
    return CIFX_INVALID_POINTER;

  int32_t                  lRet              = CIFX_NO_ERROR;
  RCX_HW_IDENTIFY_REQ_T tRcxHWIdentifyReq = {0};
  uint8_t*        pbRecvData        = NULL;
  uint32_t         ulRecvLen         = 0;

  tRcxHWIdentifyReq.tHead.ulSrc    = 0;
  tRcxHWIdentifyReq.tHead.ulDest   = RCX_PACKET_DEST_SYSTEM;
  tRcxHWIdentifyReq.tHead.ulCmd    = RCX_HW_IDENTIFY_REQ;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                            (BYTE*)&tRcxHWIdentifyReq,
                                            sizeof(tRcxHWIdentifyReq),
                                            pbRecvData,
                                            ulRecvLen,
                                            m_pcEndpoint,
                                            RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                            CIFX_INIT_TIMEOUT);

  if(CIFX_NO_ERROR == lRet)
  {
    RCX_HW_IDENTIFY_CNF_T* ptCnf = reinterpret_cast<RCX_HW_IDENTIFY_CNF_T*>(pbRecvData);

    if (~ptCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
    {
      lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
    } else if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
    {
      *ptHardwareIdentify = ptCnf->tData;
    }
    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Read device hardware information
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadHardwareInfo( RCX_HW_HARDWARE_INFO_CNF_DATA_T* ptHardwareInfo)
{
  if(NULL == ptHardwareInfo)
    return CIFX_INVALID_POINTER;

  int32_t                        lRet          = CIFX_NO_ERROR;
  RCX_HW_HARDWARE_INFO_REQ_T  tRcxHWInfoReg = {0};
  uint8_t*              pbRecvData    = NULL;
  uint32_t               ulRecvLen     = 0;

  tRcxHWInfoReg.tHead.ulSrc    = 0;
  tRcxHWInfoReg.tHead.ulDest   = RCX_PACKET_DEST_SYSTEM;
  tRcxHWInfoReg.tHead.ulCmd    = RCX_HW_HARDWARE_INFO_REQ;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                            (BYTE*)&tRcxHWInfoReg,
                                            sizeof(tRcxHWInfoReg),
                                            pbRecvData,
                                            ulRecvLen,
                                            m_pcEndpoint,
                                            RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    RCX_HW_HARDWARE_INFO_CNF_T* ptCnf = reinterpret_cast<RCX_HW_HARDWARE_INFO_CNF_T*>(pbRecvData);

    if (~ptCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
    {
      lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
    } else if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
    {
      *ptHardwareInfo = ptCnf->tData;
    }
    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Read channel information
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadChannelInfo( uint32_t ulAreaIndex, CHANNEL_INFO_T* ptChannelInfo)
{
  if(NULL == ptChannelInfo)
    return CIFX_INVALID_POINTER;

  if(NETX_MAX_SUPPORTED_CHANNELS <= ulAreaIndex)
    return CIFX_INVALID_PARAMETER;

  int32_t lRet = CIFX_NO_ERROR;

  RCX_DPM_GET_BLOCK_INFO_CNF_DATA_T tBlockInfo;
  lRet = ReadBlockInfo(ulAreaIndex, 0, &tBlockInfo);

  if( (CIFX_NO_ERROR == lRet) ||
      /* Bootloader does not support this command. Proceed anyway because there has to be the system channel. */
      ((0 == ulAreaIndex) && (RCX_E_UNKNOWN_COMMAND == lRet)) )
  {
    NETX_CHANNEL_INFO_BLOCK tChannelInfoBlock;

    lRet = ReadChannelInfoBlock(ulAreaIndex, sizeof(tChannelInfoBlock), &tChannelInfoBlock);

    /* Older bootloader may not support this command. Proceed anyway because there has to be the system channel. */
    if ((0 == ulAreaIndex) && (RCX_E_UNKNOWN_COMMAND == lRet))
    {
      /* Command not supported. Use default system channel */
      tChannelInfoBlock.tSystem.bChannelType             = RCX_CHANNEL_TYPE_SYSTEM;
      tChannelInfoBlock.tSystem.ulSizeOfChannel          = NETX_SYSTEM_CHANNEL_SIZE;
      tChannelInfoBlock.tSystem.bSizePositionOfHandshake = RCX_HANDSHAKE_POSITION_CHANNEL|RCX_HANDSHAKE_SIZE_8BIT;
      tChannelInfoBlock.tSystem.usSizeOfMailbox          = 256;
      tChannelInfoBlock.tSystem.usMailboxStartOffset     = 256;

      lRet = CIFX_NO_ERROR;
    }

    if(CIFX_NO_ERROR == lRet)
    {
      if ( (tChannelInfoBlock.tCom.bChannelType == RCX_CHANNEL_TYPE_COMMUNICATION) ||
           (tChannelInfoBlock.tCom.bChannelType == RCX_CHANNEL_TYPE_SYSTEM)   )
      {
        /* This is the system channel or a communication channel, so continue with firmware request */
        uint32_t ulChannelId = (tChannelInfoBlock.tCom.bChannelType == RCX_CHANNEL_TYPE_SYSTEM)?RCX_SYSTEM_CHANNEL:tChannelInfoBlock.tCom.bChannelId;

        /* Query Firmware name. If that works ok, we can continue enumerating blocks */
        RCX_FW_IDENTIFICATION_T tFWIdentify = {0};
        if(CIFX_NO_ERROR == (lRet = ReadChannelFirmware(ulChannelId, &tFWIdentify)))
        {
          ptChannelInfo->tFirmwareInfo  = tFWIdentify;
          ptChannelInfo->ulAreaNr       = ulAreaIndex;
          ptChannelInfo->ulChannelId    = ulChannelId;
          ptChannelInfo->ulOpenCount    = 0;
          ptChannelInfo->ulRecvCount    = 0;
          ptChannelInfo->hRecvSemaphore = ::CreateSemaphore(NULL, 0, MAXLONG, NULL);
          ptChannelInfo->ulHSKFlagSize  = (uint32_t)(tChannelInfoBlock.tCom.bSizePositionOfHandshake & RCX_HANDSHAKE_SIZE_MASK);
          ptChannelInfo->ulNumOfBlocks  = tChannelInfoBlock.tCom.bNumberOfBlocks;
          InitializeCriticalSection(&ptChannelInfo->tcsRecvLock);
        }
      } else
      {
        /* Return error as we are only interested in system and communication channel */
        lRet = CIFX_INVALID_CHANNEL;
      }
    }
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Read Channel Firmware
///   \param  ulAreaIndex   Area index
///   \param  ulSubblockIdx Subblock index
///   \param  ptBlockInfo   Reference to block info
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadBlockInfo(uint32_t ulAreaIndex, uint32_t ulSubblockIdx, RCX_DPM_GET_BLOCK_INFO_CNF_DATA_T* ptBlockInfo)
{
  if (NULL == ptBlockInfo)
    return CIFX_INVALID_PARAMETER;

  int32_t                         lRet          = CIFX_NO_ERROR;
  RCX_DPM_GET_BLOCK_INFO_REQ_T tBlockInfoReq = {0};
  uint8_t*               pbRecvData    = NULL;
  uint32_t                ulRecvLen     = 0;

  tBlockInfoReq.tHead.ulDest          = RCX_PACKET_DEST_SYSTEM;
  tBlockInfoReq.tHead.ulLen           = sizeof(tBlockInfoReq.tData);
  tBlockInfoReq.tHead.ulCmd           = RCX_DPM_GET_BLOCK_INFO_REQ;
  tBlockInfoReq.tData.ulAreaIndex     = ulAreaIndex;
  tBlockInfoReq.tData.ulSubblockIndex = ulSubblockIdx;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                            (BYTE*)&tBlockInfoReq,
                                            sizeof(tBlockInfoReq),
                                            pbRecvData,
                                            ulRecvLen,
                                            m_pcEndpoint,
                                            RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    RCX_DPM_GET_BLOCK_INFO_CNF_T* ptCnf = reinterpret_cast<RCX_DPM_GET_BLOCK_INFO_CNF_T*>(pbRecvData);

    if (~ptCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
    {
      lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
    } else if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
    {
      *ptBlockInfo = ptCnf->tData;
    }
    delete [] pbRecvData;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Read Channel Firmware
///   \param  ulChannel  Size of data buffer
///   \param  ptFirmware Reference to firmare information
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadChannelFirmware( uint32_t ulChannel, RCX_FW_IDENTIFICATION_T* ptFirmware)
{
  if (NULL == ptFirmware)
    return CIFX_INVALID_PARAMETER;

  int32_t                              lRet           = CIFX_NO_ERROR;
  RCX_FIRMWARE_IDENTIFY_REQ_T       tFWIdentifyReq = {0};
  uint8_t*                    pbRecvData     = NULL;
  uint32_t                     ulRecvLen      = 0;

  tFWIdentifyReq.tHead.ulSrc       = 0;
  tFWIdentifyReq.tHead.ulLen       = sizeof(tFWIdentifyReq.tData);
  tFWIdentifyReq.tHead.ulCmd       = RCX_FIRMWARE_IDENTIFY_REQ;
  tFWIdentifyReq.tData.ulChannelId = ulChannel;


  /* Query Firmware name. If that works ok, we can continue enumerating blocks */
  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                            (BYTE*)&tFWIdentifyReq,
                                            sizeof(tFWIdentifyReq),
                                            pbRecvData,
                                            ulRecvLen,
                                            m_pcEndpoint,
                                            RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    RCX_FIRMWARE_IDENTIFY_CNF_T* ptCnf = reinterpret_cast<RCX_FIRMWARE_IDENTIFY_CNF_T*>(pbRecvData);

    if (~ptCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
    {
      lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
    } else if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
    {
      *ptFirmware = ptCnf->tData.tFirmwareIdentification;
    }
    delete [] pbRecvData;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Read System Information
///   \param  ptSystemInfo Reference to system info
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadSysInformation( SYSTEM_CHANNEL_SYSTEM_INFORMATION* ptSystemInfo)
{
  if (NULL == ptSystemInfo)
    return CIFX_INVALID_PARAMETER;

  ptSystemInfo->ulSystemError   =   CIFX_NO_ERROR;
  ptSystemInfo->ulDpmTotalSize  =   m_tSystemInfo.ulDpmTotalSize;
  ptSystemInfo->ulMBXSize       =   NETX_SYSTEM_MAILBOX_MIN_SIZE;
  ptSystemInfo->ulDeviceNumber  =   m_tDeviceInfo.ulDeviceNumber;
  ptSystemInfo->ulSerialNumber  =   m_tDeviceInfo.ulSerialNumber;

  EnterCriticalSection(&m_tcsChannels);
  if(m_cmChannels.size() > 0)
    ptSystemInfo->ulOpenCnt = m_cmChannels[RCXPACKET_SYSTEMCHANNEL_INTERNAL].ulOpenCount;
  else
    ptSystemInfo->ulOpenCnt = 0;
  LeaveCriticalSection(&m_tcsChannels);

  return CIFX_NO_ERROR;
}

/////////////////////////////////////////////////////////////////////////////
/// Read System InformationBlock
///   \param  ptSysInfoBlock  Reference to system info block
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadSysInfoBlock( NETX_SYSTEM_INFO_BLOCK* ptSysInfoBlock)
{
  if (NULL == ptSysInfoBlock)
    return CIFX_INVALID_PARAMETER;

  int32_t                            lRet                  = CIFX_NO_ERROR;
  RCX_READ_SYS_INFO_BLOCK_REQ_T   tRcxSysInfoBlockReq   = {0};
  uint8_t*                  pbRecvData            = NULL;
  uint32_t                   ulRecvLen             = 0;

  tRcxSysInfoBlockReq.tHead.ulSrc    = 0;
  tRcxSysInfoBlockReq.tHead.ulDest   = RCX_PACKET_DEST_SYSTEM;
  tRcxSysInfoBlockReq.tHead.ulCmd    = RCX_SYSTEM_INFORMATION_BLOCK_REQ;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                            (BYTE*)&tRcxSysInfoBlockReq,
                                            sizeof(tRcxSysInfoBlockReq),
                                            pbRecvData,
                                            ulRecvLen,
                                            m_pcEndpoint,
                                            RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    RCX_READ_SYS_INFO_BLOCK_CNF_T* ptCnf = reinterpret_cast<RCX_READ_SYS_INFO_BLOCK_CNF_T*>(pbRecvData);

    if (~ptCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
    {
      lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
    } else
    {
      lRet = ptCnf->tHead.ulSta;
      if(CIFX_NO_ERROR == lRet)
      {
        *ptSysInfoBlock = ptCnf->tData.tSystemInfo;
      } else if (RCX_E_UNKNOWN_COMMAND == lRet)
      {
        lRet = CIFX_NO_ERROR;
        ptSysInfoBlock->ulDeviceNumber   = m_tDeviceInfo.ulDeviceNumber;
        ptSysInfoBlock->ulSerialNumber   = m_tDeviceInfo.ulSerialNumber;
        ptSysInfoBlock->ausHwOptions[0]  = m_tDeviceInfo.ausHwOptions[0];
        ptSysInfoBlock->ausHwOptions[1]  = m_tDeviceInfo.ausHwOptions[1];
        ptSysInfoBlock->ausHwOptions[2]  = m_tDeviceInfo.ausHwOptions[2];
        ptSysInfoBlock->ausHwOptions[3]  = m_tDeviceInfo.ausHwOptions[3];
        ptSysInfoBlock->usDeviceClass    = m_tDeviceInfo.usDeviceClass;
        ptSysInfoBlock->bHwRevision      = m_tDeviceInfo.bHwRevision;
        ptSysInfoBlock->bHwCompatibility = m_tDeviceInfo.bHwCompatibility;
        ptSysInfoBlock->usManufacturer   = RCX_MANUFACTURER_HILSCHER_GMBH;
        memcpy((char*)ptSysInfoBlock->abCookie,CIFX_DPMSIGNATURE_BSL_STR, 4);
      }
    }
    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Read System Channel Block
///   \param  ulChannel Channel number
///   \param  ulSize    Size of data buffer
///   \param  pvData    Reference of data buffer
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadChannelInfoBlock( uint32_t ulChannel, uint32_t ulSize, void* pvData)
{
  int32_t                              lRet                  = CIFX_NO_ERROR;
  RCX_READ_CHNL_INFO_BLOCK_REQ_T    tRcxSysChnBlockReq    = {0};
  uint8_t*                    pbRecvData            = NULL;
  uint32_t                     ulRecvLen             = 0;

  if( ulSize < (uint32_t)sizeof(RCX_READ_CHNL_INFO_BLOCK_CNF_DATA_T))
  {
    lRet = CIFX_INVALID_BUFFERSIZE;
  } else
  {

    tRcxSysChnBlockReq.tHead.ulSrc       = 0;
    tRcxSysChnBlockReq.tHead.ulDest      = RCX_PACKET_DEST_SYSTEM;
    tRcxSysChnBlockReq.tHead.ulLen       = sizeof(tRcxSysChnBlockReq.tData);
    tRcxSysChnBlockReq.tHead.ulCmd       = RCX_CHANNEL_INFORMATION_BLOCK_REQ;
    tRcxSysChnBlockReq.tData.ulChannelId = ulChannel;

    lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                              (BYTE*)&tRcxSysChnBlockReq,
                                              sizeof(tRcxSysChnBlockReq),
                                              pbRecvData,
                                              ulRecvLen,
                                              m_pcEndpoint,
                                              RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                              CIFX_TO_CONT_PACKET);

    if(CIFX_NO_ERROR == lRet)
    {
      RCX_READ_CHNL_INFO_BLOCK_CNF_T* ptCnf = reinterpret_cast<RCX_READ_CHNL_INFO_BLOCK_CNF_T*>(pbRecvData);

      if (~ptCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
      {
        lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
      } else if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
      {
        uint32_t ulCopySize = ptCnf->tHead.ulLen;

        if(ulSize < ulCopySize)
        {
          ulCopySize = ulSize;
          lRet = CIFX_BUFFER_TOO_SHORT;
        }
        memcpy(pvData, &ptCnf->tData, ulCopySize);
      }
      delete [] pbRecvData;
    }
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Read System Control Block
///   \param  ptSysControlBlock Reference to system control block
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadSysControlBlock( NETX_SYSTEM_CONTROL_BLOCK* ptSysControlBlock)
{
  if (NULL == ptSysControlBlock)
    return CIFX_INVALID_PARAMETER;

  int32_t                              lRet                  = CIFX_NO_ERROR;
  RCX_READ_SYS_CNTRL_BLOCK_REQ_T    tRcxSysCtlBlockReq    = {0};
  uint8_t*                    pbRecvData            = NULL;
  uint32_t                     ulRecvLen             = 0;

  tRcxSysCtlBlockReq.tHead.ulSrc    = 0;
  tRcxSysCtlBlockReq.tHead.ulDest   = RCX_PACKET_DEST_SYSTEM;
  tRcxSysCtlBlockReq.tHead.ulCmd    = RCX_SYSTEM_CONTROL_BLOCK_REQ;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                            (BYTE*)&tRcxSysCtlBlockReq,
                                            sizeof(tRcxSysCtlBlockReq),
                                            pbRecvData,
                                            ulRecvLen,
                                            m_pcEndpoint,
                                            RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    RCX_READ_SYS_CNTRL_BLOCK_CNF_T* ptCnf = reinterpret_cast<RCX_READ_SYS_CNTRL_BLOCK_CNF_T*>(pbRecvData);

    if (~ptCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
    {
      lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
    } else if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
    {
      *ptSysControlBlock = ptCnf->tData.tSystemControl;
    }
    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Read System Status Block
///   \param  ptSysStatusBlock Reference to system status block
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadSysStatusBlock( NETX_SYSTEM_STATUS_BLOCK* ptSysStatusBlock)
{
  if (NULL == ptSysStatusBlock)
    return CIFX_INVALID_PARAMETER;

  int32_t                              lRet                  = CIFX_NO_ERROR;
  RCX_READ_SYS_STATUS_BLOCK_REQ_T   tRcxSysStaBlockReq    = {0};
  uint8_t*                    pbRecvData            = NULL;
  uint32_t                     ulRecvLen             = 0;

  tRcxSysStaBlockReq.tHead.ulSrc    = 0;
  tRcxSysStaBlockReq.tHead.ulDest   = RCX_PACKET_DEST_SYSTEM;
  tRcxSysStaBlockReq.tHead.ulCmd    = RCX_SYSTEM_STATUS_BLOCK_REQ;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                            (BYTE*)&tRcxSysStaBlockReq,
                                            sizeof(tRcxSysStaBlockReq),
                                            pbRecvData,
                                            ulRecvLen,
                                            m_pcEndpoint,
                                            RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    RCX_READ_SYS_STATUS_BLOCK_CNF_T* ptCnf = reinterpret_cast<RCX_READ_SYS_STATUS_BLOCK_CNF_T*>(pbRecvData);

    if (~ptCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
    {
      lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
    } else if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
    {
      *ptSysStatusBlock  = ptCnf->tData.tSystemState;
    }
    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Read Com Flags
///   \param  ulAreaIndex Area index
///   \param  pulNetxComFlag Reference to netx com flags
///   \param  pulHostComFlag Reference to host com flags
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadComFlags( uint32_t ulAreaIndex, uint32_t* pulNetxComFlag, uint32_t* pulHostComFlag)
{
  if ( (NULL == pulNetxComFlag) ||
       (NULL == pulHostComFlag)   )
    return CIFX_INVALID_PARAMETER;

  int32_t                           lRet              = CIFX_NO_ERROR;
  RCX_DPM_GET_COMFLAG_INFO_REQ_T tRcxComFlagReq    = {0};
  uint8_t*                 pbRecvData        = NULL;
  uint32_t                  ulRecvLen         = 0;

  tRcxComFlagReq.tHead.ulSrc       = 0;
  tRcxComFlagReq.tHead.ulDest      = RCX_PACKET_DEST_SYSTEM;
  tRcxComFlagReq.tHead.ulCmd       = RCX_DPM_GET_COMFLAG_INFO_REQ;
  tRcxComFlagReq.tHead.ulLen       = sizeof(tRcxComFlagReq.tData.ulAreaIndex);
  tRcxComFlagReq.tData.ulAreaIndex = ulAreaIndex;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                            (BYTE*)&tRcxComFlagReq,
                                            sizeof(tRcxComFlagReq),
                                            pbRecvData,
                                            ulRecvLen,
                                            m_pcEndpoint,
                                            RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    RCX_DPM_GET_COMFLAG_INFO_CNF_T* ptCnf = reinterpret_cast<RCX_DPM_GET_COMFLAG_INFO_CNF_T*>(pbRecvData);

    if (~ptCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
    {
      lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
    } else if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
    {
      *pulNetxComFlag = ptCnf->tData.ulNetxComFlag;
      *pulHostComFlag = ptCnf->tData.ulHostComFlag;
    }
    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Read Channel Control Block
///   \param  ulChannel      Channel number
///   \param  ptControlBlock Reference to control block
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadChannelControlBlock( uint32_t ulChannel, NETX_CONTROL_BLOCK* ptControlBlock)
{
  if (NULL == ptControlBlock)
    return CIFX_INVALID_PARAMETER;

  int32_t                              lRet                  = CIFX_NO_ERROR;
  RCX_READ_COMM_CNTRL_BLOCK_REQ_T   tRcxCommonControlReq  = {0};
  uint8_t*                    pbRecvData            = NULL;
  uint32_t                     ulRecvLen             = 0;

  tRcxCommonControlReq.tHead.ulSrc       = 0;
  tRcxCommonControlReq.tHead.ulDest      = RCX_PACKET_DEST_SYSTEM;
  tRcxCommonControlReq.tHead.ulCmd       = RCX_CONTROL_BLOCK_REQ;
  tRcxCommonControlReq.tHead.ulLen       = sizeof(tRcxCommonControlReq.tData.ulChannelId);
  tRcxCommonControlReq.tData.ulChannelId = ulChannel;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                            (BYTE*)&tRcxCommonControlReq,
                                            sizeof(tRcxCommonControlReq),
                                            pbRecvData,
                                            ulRecvLen,
                                            m_pcEndpoint,
                                            RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    RCX_READ_COMM_CNTRL_BLOCK_CNF_T* ptCnf = reinterpret_cast<RCX_READ_COMM_CNTRL_BLOCK_CNF_T*>(pbRecvData);

    if (~ptCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
    {
      lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
    } else if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
    {
      *ptControlBlock = ptCnf->tData.tControl;
    }
    delete [] pbRecvData;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Read Common Status Block
///   \param  ulChannel      Channel number
///   \param  ptStatusBlock  Reference to common status block
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadCommonStatusBlock( uint32_t ulChannel, NETX_COMMON_STATUS_BLOCK* ptStatusBlock)
{
  if (NULL == ptStatusBlock)
    return CIFX_INVALID_PARAMETER;

  int32_t                              lRet                  = CIFX_NO_ERROR;
  RCX_READ_COMMON_STS_BLOCK_REQ_T   tRcxCommonStatusReq   = {0};
  uint8_t*                    pbRecvData            = NULL;
  uint32_t                     ulRecvLen             = 0;

  tRcxCommonStatusReq.tHead.ulSrc       = 0;
  tRcxCommonStatusReq.tHead.ulDest      = RCX_PACKET_DEST_SYSTEM;
  tRcxCommonStatusReq.tHead.ulCmd       = RCX_DPM_GET_COMMON_STATE_REQ;
  tRcxCommonStatusReq.tHead.ulLen       = sizeof(tRcxCommonStatusReq.tData.ulChannelId);
  tRcxCommonStatusReq.tData.ulChannelId = ulChannel;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                            (BYTE*)&tRcxCommonStatusReq,
                                            sizeof(tRcxCommonStatusReq),
                                            pbRecvData,
                                            ulRecvLen,
                                            m_pcEndpoint,
                                            RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    RCX_READ_COMMON_STS_BLOCK_CNF_T* ptCnf = reinterpret_cast<RCX_READ_COMMON_STS_BLOCK_CNF_T*>(pbRecvData);

    if (~ptCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
    {
      lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
    } else if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
    {
      *ptStatusBlock = ptCnf->tData.tCommonStatus;
    }
    delete [] pbRecvData;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Read DPM IO Information
///   \param  ulChannel      Channel number
///   \param  ptStatusBlock  Reference to common status block
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::ReadDPMIOInfo( CIFXHANDLE  hDevice, CHANNEL_INFORMATION* ptChannelInfo)
{
  if(NULL == ptChannelInfo)
    return CIFX_INVALID_POINTER;

  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  /* Get Device and Channel number from handle */
  if(!InterpretHandle((uint32_t) hDevice, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  int32_t             lRet              = CIFX_NO_ERROR;
  RCX_PACKET_HEADER   tRcxDPMIOInfoReq  = {0};
  uint8_t*            pbRecvData        = NULL;
  uint32_t            ulRecvLen         = 0;

  tRcxDPMIOInfoReq.ulSrc       = 0;
  tRcxDPMIOInfoReq.ulDest      = RCX_PACKET_DEST_DEFAULT_CHANNEL;
  tRcxDPMIOInfoReq.ulCmd       = RCX_GET_DPM_IO_INFO_REQ;
  tRcxDPMIOInfoReq.ulLen       = 0;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                            (BYTE*)&tRcxDPMIOInfoReq,
                                            sizeof(tRcxDPMIOInfoReq),
                                            pbRecvData,
                                            ulRecvLen,
                                            m_pcEndpoint,
                                            bChannel,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    RCX_GET_DPM_IO_INFO_CNF_T* ptCnf = reinterpret_cast<RCX_GET_DPM_IO_INFO_CNF_T*>(pbRecvData);

    if (~ptCnf->tHead.ulCmd & RCX_MSK_PACKET_ANSWER)
    {
      lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
    } else if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
    {
      ptChannelInfo->ulIOInAreaCnt = ptCnf->tData.ulNumIOBlockInfo;
      ptChannelInfo->ulIOOutAreaCnt = ptCnf->tData.ulNumIOBlockInfo;
    }
    delete [] pbRecvData;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Build channel handle
///   \param  bBoard    Board number (byte)
///   \param  bChannel  Channel number (byte)
///   \return ulHandle  uint32_t version of the new handle
/////////////////////////////////////////////////////////////////////////////
uint32_t CrcXPacket::BuildHandle( BYTE bBoard, BYTE bChannel)
{
  uint32_t ulHandle = 0;

  ulHandle  = (uint32_t)(( *((uint32_t*)this)) << 16);
  ulHandle ^= (uint32_t)((bBoard)  << 8);
  ulHandle ^= (uint32_t)(bChannel);

  return ulHandle;
}

/////////////////////////////////////////////////////////////////////////////
/// Interpret channel handle
///   \param  ulHandle  Handle to interpret
///   \param  bBoard    Returns board number (byte)
///   \param  bChannel  Returns channel number (byte)
///   \return TRUE if handle is valid, FALSE otherwise
/////////////////////////////////////////////////////////////////////////////
bool CrcXPacket::InterpretHandle( uint32_t ulHandle,
                                  BYTE&         bBoard,
                                  BYTE&         bChannel)
{
  bool fRet = false;

  if( ((*((uint32_t*)this)) << 16) == (ulHandle & 0xFFFF0000))
  {
    bBoard    = (BYTE) (( ulHandle & 0x0000FF00) >> 8);
    bChannel  = (BYTE)  ( ulHandle & 0x000000FF);

   fRet = true;
  }
  return fRet;
}

//
/////////////////////////////////////////////////////////////////////////////////////////////
/// Destroy saved connection
///   \param  hDevice     Handle of device
///   \param  ulResetMode CIFX_SYSTEMSTART or CIFX_CHANNELINIT
///   \param  ulTimeout   Reset timeout
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////////////////////
int32_t  CrcXPacket::Reset(HANDLE        hDevice,
                        uint32_t ulResetMode,
                        uint32_t ulTimeout)

{
  if(NULL == hDevice)
    return CIFX_INVALID_HANDLE;

  int32_t                      lRet            = CIFX_INVALID_COMMAND;
  /* Get Device and Channel number from handle */
  BYTE                      bBoard          = 0;
  BYTE                      bChannel        = 0;
  RCX_PACKET                tPacket         = {0};

  // Values returned by transfer packet function
  uint8_t*            pbRecvData      = NULL;
  uint32_t             ulRecvLen       = 0;

  if(!InterpretHandle((uint32_t) hDevice, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  switch(ulResetMode)
  {
    case CIFX_SYSTEMSTART:
    {
      RCX_FIRMWARE_RESET_REQ_T* ptReset = (RCX_FIRMWARE_RESET_REQ_T*)&tPacket;

      ptReset->tHead.ulSrc  = 0;
      ptReset->tHead.ulDest = RCX_PACKET_DEST_SYSTEM;
      ptReset->tHead.ulLen  = sizeof(RCX_FIRMWARE_RESET_REQ_DATA_T);
      ptReset->tHead.ulCmd  = RCX_FIRMWARE_RESET_REQ;

      ptReset->tData.ulTimeToReset = 500; /* Device shall wait 500ms to proceed with reset */
      ptReset->tData.ulResetMode   = 0;

      /* We need to send a systemstart to systemchannel */
      bChannel = RCXPACKET_SYSTEMCHANNEL_INTERNAL;

      lRet = CIFX_NO_ERROR;
    }
    break;

    case CIFX_CHANNELINIT:
      if(bChannel == RCXPACKET_SYSTEMCHANNEL_INTERNAL)
      {
        lRet = CIFX_INVALID_CHANNEL;
      } else
      {
        tPacket.tHeader.ulSrc  = 0;
        tPacket.tHeader.ulDest = RCX_PKT_COMM_CHANNEL_TOKEN;
        tPacket.tHeader.ulCmd  = RCX_CHANNEL_INIT_REQ;

        lRet = CIFX_NO_ERROR;
      }
    break;

    default:
      lRet = CIFX_INVALID_COMMAND;
    break;
  }

  if(CIFX_NO_ERROR == lRet)
  {
    lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                              (BYTE*)&tPacket,
                                              sizeof(tPacket.tHeader) + tPacket.tHeader.ulLen,
                                              pbRecvData,
                                              ulRecvLen,
                                              m_pcEndpoint,
                                              bChannel,
                                              ulTimeout);


    // Send rcx packet -> reset, depends on given parameter
    if(CIFX_NO_ERROR == lRet)
    {
      RCX_FIRMWARE_RESET_CNF_T* ptCnf = reinterpret_cast<RCX_FIRMWARE_RESET_CNF_T*>(pbRecvData);

      if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
      {
        // System start is done, so handle host side
        // -> Host side will be handled about cifXfunction implementation
      }
      delete  pbRecvData;
    }
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdevice functions


/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceOpen, open a specific sysdevice channel
///   \param hDriver      Handle of driver
///   \param szBoard      Board name of the device
///   \param phSysdevice  Reference of the Sysdevice handle
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xSysdeviceOpen(CIFXHANDLE  hDriver,
                                char*       szBoard,
                                CIFXHANDLE* phSysdevice)
{
  UNREFERENCED_PARAMETER(hDriver);

  /* NOTE: Currently a serial device always has only one "Device" */
  /*       As long as we are running, we should have board. Otherwise */
  /*       no rcXPacket object exists */
  /* NOTE: Channel handling realized by using of m_cvChannel (control structure) */

  int32_t          lRet          = CIFX_NO_ERROR;
  std::string   sBoard(szBoard);
  std::string   sReferenceName(NX_DEFAULT_NAME);
  std::string   sBoardNumber;
  int32_t          lBoardnumber;

  /* Check Board name is valid */
  if( std::string::npos == sBoard.find_first_of( NX_DEFAULT_NAME))
    lRet =  CIFX_INVALID_BOARD;

  if(CIFX_NO_ERROR == lRet)
  {
    /* Extract the device number from the board name */
    sBoardNumber = sBoard.substr(sReferenceName.length());
    lBoardnumber = atol(sBoardNumber.c_str());

    /* Extract the device number from class object device name */
    if(std::string::npos == m_szDeviceName.find(NX_DEFAULT_NAME))
      lRet = CIFX_INVALID_BOARD;

    if(CIFX_NO_ERROR == lRet)
    {
      sBoardNumber = m_szDeviceName.substr(sReferenceName.length());

      /* Check if we have such a board */
      if( (lBoardnumber != atol(sBoardNumber.c_str())) ||
          !m_fConnectionValid                        )
        lRet = CIFX_INVALID_BOARD;

      /*
        HOW TO BUILD A HANDLE

        |--------|--------|--------|--------|
        | lower 16 bit of | Board  | Channel|
        | current this    | number | number |
        | pointer         |        |        |
        |--------|--------|--------|--------|

        NOTE: Sysdevice -> part of channnel number 0xFF
      */
      if(CIFX_NO_ERROR == lRet)
      {
        EnterCriticalSection(&m_tcsChannels);

        CHANNEL_MAP::iterator iter = m_cmChannels.find(RCXPACKET_SYSTEMCHANNEL_INTERNAL);

        if(iter == m_cmChannels.end())
        {
          lRet = CIFX_INVALID_BOARD;
        } else
        {
          *phSysdevice = (CIFXHANDLE)BuildHandle( (BYTE)lBoardnumber,
                                                  RCXPACKET_SYSTEMCHANNEL_INTERNAL);
          iter->second.ulOpenCount++;
        }

        LeaveCriticalSection(&m_tcsChannels);

        InterlockedIncrement(&m_lRefCount);
      }
    }
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceClose, close a specific sysdevice channel
///   \param hSysdevice       Handle of the system device
///   \return CIFX_NO_ERROR  on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xSysdeviceClose( CIFXHANDLE  hSysdevice)
{
  BYTE bBoard   = 0;
  BYTE bChannel = 0;
  int32_t lRet     = CIFX_NO_ERROR;

  if(!InterpretHandle((uint32_t)hSysdevice, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  /* Note:  we supporting only one board */
  if( (0 != bBoard)                          ||
      (RCXPACKET_SYSTEMCHANNEL_INTERNAL != bChannel) )
  {
    lRet =  CIFX_INVALID_HANDLE;
  } else
  {
    InterlockedDecrement(&m_lRefCount);

    EnterCriticalSection(&m_tcsChannels);

    CHANNEL_MAP::iterator iter = m_cmChannels.find(bChannel);

    if(iter != m_cmChannels.end())
    {
      CHANNEL_INFO_T& tChannel = iter->second;

      if(0 == tChannel.ulOpenCount)
      {
        lRet = CIFX_DRV_NOT_INITIALIZED;

      } else if(0 == --tChannel.ulOpenCount)
      {
        /* Last one closed the system device, so cleanup queues */

        EnterCriticalSection(&tChannel.tcsRecvLock);

        while(tChannel.cvRecvQueue.size() > 0)
        {
          PHIL_TRANSPORT_PACKET ptPacket = tChannel.cvRecvQueue.front();
          CTransportLayer::FreeTransportFrame(ptPacket);
          tChannel.cvRecvQueue.pop_front();
        }
        LeaveCriticalSection(&tChannel.tcsRecvLock);
      }
    }

    LeaveCriticalSection(&m_tcsChannels);
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceGetMBXState, get mail box state
///   \param hSysdevice       handle, of a sysdevice channel
///   \param pulRecvPktCount  pointer to a receive packet counter
///   \param pulSendPktCount  pointer to a send packet counter
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xSysdeviceGetMBXState(CIFXHANDLE      hSysdevice,
                                       uint32_t*  pulRecvPktCount,
                                       uint32_t*  pulSendPktCount)
{
  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  /* Get Device and Channel number from handle */
  if( (!InterpretHandle((uint32_t) hSysdevice, bBoard, bChannel)) ||
      (RCXPACKET_SYSTEMCHANNEL_INTERNAL!= bChannel)                      )
    return CIFX_INVALID_HANDLE;

  int32_t lRet = CIFX_NO_ERROR;

  CHANNEL_MAP::iterator iter = m_cmChannels.find(bChannel);

  if(iter == m_cmChannels.end())
  {
    lRet = CIFX_INVALID_HANDLE;
  } else
  {
    CHANNEL_INFO_T& tChannel = iter->second;

    EnterCriticalSection(&tChannel.tcsRecvLock);

    *pulRecvPktCount = (uint32_t)tChannel.cvRecvQueue.size();
    if(!m_pcTransportLayer->GetSendMbxState( 0, *pulSendPktCount))
      *pulSendPktCount = 0;

    LeaveCriticalSection(&tChannel.tcsRecvLock);

    lRet = CIFX_NO_ERROR;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdevicePutPacket, send packet
///   \param hSysdevice       handle, of a sysdevice channel
///   \param ptSendPkt        pointer to a send packet
///   \param ulTimeout        Timeout
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xSysdevicePutPacket( CIFXHANDLE    hSysdevice,
                                      CIFX_PACKET*  ptSendPkt,
                                      uint32_t ulTimeout)
{
  /* We have a rcX packet and only one connection to one device */
  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  /* Get Device and Channel number from handle */
  if( !InterpretHandle((uint32_t)hSysdevice, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  int32_t lRet = CIFX_NO_ERROR;

  CHANNEL_MAP::iterator iter = m_cmChannels.find(bChannel);

  if(iter == m_cmChannels.end())
  {
    lRet = CIFX_INVALID_HANDLE;
  } else
  {
    HIL_TRANSPORT_PACKET tPacket = {0};

    m_pcTransportLayer->GeneratePacketHeader(&tPacket.tHeader);

    tPacket.tHeader.usDataType  = HIL_TRANSPORT_TYPE_RCX_PACKET;
    tPacket.tHeader.ulLength    = ptSendPkt->tHeader.ulLen + CIFX_PACKET_HEADER_SIZE;
    tPacket.tHeader.bChannel    = bChannel;
    tPacket.pbData              = (uint8_t*) ptSendPkt;

    lRet = m_pcTransportLayer->SendPacket( &tPacket,
                                           NULL, /* We want to make this synchrounus (Layer = NULL) */
                                           ulTimeout);

    CHANNEL_INFO_T& tChannelInfo = iter->second;
    tChannelInfo.ulSendPacketCnt++;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceGetPacket, get received packets
///   \param hSysdevice       Handle, of a sysdevice channel
///   \param ulSize           Size of buffer
///   \param ptRecvPkt        Pointer to received packet
///   \param ulTimeout        Timeout
///   \return CIFX_NO_ERROR of success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xSysdeviceGetPacket( CIFXHANDLE    hSysdevice,
                                      uint32_t ulSize,
                                      CIFX_PACKET*  ptRecvPkt,
                                      uint32_t ulTimeout)
 {
  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  /* Get Device and Channel number from handle */
  if(!InterpretHandle((uint32_t) hSysdevice, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  CHANNEL_MAP::iterator iter = m_cmChannels.find(bChannel);
  int32_t                  lRet = CIFX_NO_ERROR;

  if(iter == m_cmChannels.end())
  {
    lRet = CIFX_INVALID_HANDLE;
  } else
  {
    CHANNEL_INFO_T& tChannel = iter->second;

    DWORD dwWaitRes = ::WaitForSingleObject(tChannel.hRecvSemaphore, ulTimeout);

    switch(dwWaitRes)
    {
      case WAIT_OBJECT_0:
      {
        HIL_TRANSPORT_PACKET* ptHilPacket = NULL;

        EnterCriticalSection(&tChannel.tcsRecvLock);

        if(tChannel.cvRecvQueue.empty())
        {
          // we do not expect to have no message at this point because we are signaled
          // leave the handling
          // _ASSERT(false);
          LeaveCriticalSection(&tChannel.tcsRecvLock);
          break;
        }

        // Get the first packet of receive queue and erase queue entry
        ptHilPacket = tChannel.cvRecvQueue.front();
        tChannel.cvRecvQueue.pop_front();

        LeaveCriticalSection(&tChannel.tcsRecvLock);

        _ASSERT(ptHilPacket);

        uint32_t ulCopySize = ptHilPacket->tHeader.ulLength;
        if(ulSize < ulCopySize)
        {
          ulCopySize = ulSize;
          lRet = CIFX_BUFFER_TOO_SHORT;
        }

        memcpy(ptRecvPkt, ptHilPacket->pbData, ulCopySize);

        /* Free allocated memory, cause we returned on Receive data false, so the datalayer has to delete the packet */
        CTransportLayer::FreeTransportFrame(ptHilPacket);

        CHANNEL_INFO_T& tChannelInfo = iter->second;
        tChannelInfo.ulRecvPacketCnt++;
      }
      break;

      case WAIT_TIMEOUT:
        lRet = CIFX_DEV_GET_NO_PACKET;
      break;

      default:
      _ASSERT(false);
      break;
    }
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceInfo, get sysdevice informationen
///   \param hSysdevice       handle, of a sysdevice channel
///   \param ulCmd            command to choose which information are needed
///   \param ulSize           Size of buffer
///   \param pvInfo           pointer to buffer
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xSysdeviceInfo(CIFXHANDLE    hSysdevice,
                                uint32_t ulCmd,
                                uint32_t ulSize,
                                void*         pvInfo)
{
  UNREFERENCED_PARAMETER(hSysdevice);

  int32_t lRet = CIFX_NO_ERROR;

  switch(ulCmd)
  {
    case CIFX_INFO_CMD_SYSTEM_INFORMATION:
      if (ulSize < sizeof(SYSTEM_CHANNEL_SYSTEM_INFORMATION))
      {
        lRet = CIFX_INVALID_BUFFERSIZE;
      } else
      {
        SYSTEM_CHANNEL_SYSTEM_INFORMATION* ptSystemInfo = (SYSTEM_CHANNEL_SYSTEM_INFORMATION*)pvInfo;
        lRet = ReadSysInformation( ptSystemInfo);
      }
      break;

    case CIFX_INFO_CMD_SYSTEM_INFO_BLOCK:
      if (ulSize < sizeof(NETX_SYSTEM_INFO_BLOCK))
      {
        lRet = CIFX_INVALID_BUFFERSIZE;
      } else
      {
        NETX_SYSTEM_INFO_BLOCK* ptSysInfoBlock = (NETX_SYSTEM_INFO_BLOCK*)pvInfo;
        lRet = ReadSysInfoBlock( ptSysInfoBlock);
      }
      break;

    case CIFX_INFO_CMD_SYSTEM_CHANNEL_BLOCK:
      if( ulSize < sizeof(SYSTEM_CHANNEL_CHANNEL_INFO_BLOCK))
      {
        lRet = CIFX_INVALID_BUFFERSIZE;
      } else
      {
        SYSTEM_CHANNEL_CHANNEL_INFO_BLOCK tSystemChannelInfoBlock = {0};
        for( uint32_t ulChannelId = 0; (ulChannelId < CIFX_MAX_NUMBER_OF_CHANNEL_DEFINITION); ulChannelId++)
        {
          lRet = ReadChannelInfoBlock( ulChannelId, sizeof(tSystemChannelInfoBlock.abInfoBlock[ulChannelId]), &tSystemChannelInfoBlock.abInfoBlock[ulChannelId]);
          if (lRet == RCX_E_UNKNOWN_COMMAND)
          {
            lRet = CIFX_NO_ERROR;
          }
        }
        memcpy(pvInfo, &tSystemChannelInfoBlock, sizeof(tSystemChannelInfoBlock));
      }
      break;

    case CIFX_INFO_CMD_SYSTEM_CONTROL_BLOCK:
      if (ulSize < sizeof(NETX_SYSTEM_CONTROL_BLOCK))
      {
        lRet = CIFX_INVALID_BUFFERSIZE;
      } else
      {
        NETX_SYSTEM_CONTROL_BLOCK* ptSysContolBlock = (NETX_SYSTEM_CONTROL_BLOCK*)pvInfo;
        lRet = ReadSysControlBlock( ptSysContolBlock);
      }
      break;

    case CIFX_INFO_CMD_SYSTEM_STATUS_BLOCK:
      if (ulSize < sizeof(NETX_SYSTEM_STATUS_BLOCK))
      {
        lRet = CIFX_INVALID_BUFFERSIZE;
      } else
      {
        NETX_SYSTEM_STATUS_BLOCK* ptSysStatusBlock = (NETX_SYSTEM_STATUS_BLOCK*)pvInfo;
        lRet = ReadSysStatusBlock( ptSysStatusBlock);
      }
      break;

    default:
      lRet = CIFX_INVALID_COMMAND;
      break;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceFindFirstFile, find first file in file system
///   \param hSysdevice         Handle, of a system channel
///   \param ulChannel          Channel number
///   \param ptDirectoryInfo    Directory info
///   \param pfnRecvPktCallback Packet callback
///   \param pvUser             User data for callback
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xSysdeviceFindFirstFile(CIFXHANDLE             hSysdevice,
                                         uint32_t          ulChannel,
                                         CIFX_DIRECTORYENTRY*   ptDirectoryInfo,
                                         PFN_RECV_PKT_CALLBACK  pfnRecvPktCallback,
                                         void*                  pvUser)
{
  UNREFERENCED_PARAMETER(pfnRecvPktCallback);
  UNREFERENCED_PARAMETER(pvUser);

  // NOW, We have a rcX packet and on this time only one connection via one device
  // NOTE: We ignore the RecvCallback functions pointer, cause we don't receive a async packet

  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  /* Get Device and Channel number from handle */
  if(!InterpretHandle((uint32_t) hSysdevice, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  int32_t lRet = CIFX_NO_ERROR;

  CIFX_PACKET           tcifXPkt      = {0};
  RCX_DIR_LIST_REQ_T*   ptDirListReq  = (RCX_DIR_LIST_REQ_T*)&tcifXPkt;
  uint8_t*        pbRecvData    = NULL;
  uint32_t         ulRecvLen     = 0;

  // Check size of filename (ptDirectoryInfo->szfilename)
  if( strlen(ptDirectoryInfo->szFilename) > 0)
  {
    uint16_t usDirNameLength = (uint16_t)(strlen(ptDirectoryInfo->szFilename) + 1);

    ptDirListReq->tData.usDirNameLength = usDirNameLength;
    strncpy((char*)((&ptDirListReq->tData) + 1),
                ptDirectoryInfo->szFilename,
                usDirNameLength);
  }

  ptDirListReq->tHead.ulDest      = RCX_PACKET_DEST_SYSTEM;
  //ptDirListReq->tHead.ulSrc       = (uint32_t)hSysdevice;
  ptDirListReq->tHead.ulSrc       = 0;
  ptDirListReq->tHead.ulCmd       = RCX_DIR_LIST_REQ;
  ptDirListReq->tHead.ulLen       = (sizeof(ptDirListReq->tData) + ptDirListReq->tData.usDirNameLength) ;
  ptDirListReq->tData.ulChannelNo = (uint32_t) ulChannel;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                            (BYTE*)ptDirListReq,
                                            sizeof(*ptDirListReq) + ptDirListReq->tData.usDirNameLength,
                                            pbRecvData,
                                            ulRecvLen,
                                            m_pcEndpoint,
                                            RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    RCX_DIR_LIST_CNF_T* ptDirListCnf = reinterpret_cast<RCX_DIR_LIST_CNF_T*>(pbRecvData);

    if (~ptDirListCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
    {
      lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
    } else if( RCX_S_OK == (lRet = ptDirListCnf->tHead.ulSta) )
    {
      /* TODO: Store handle for directory list, which needs to be set by firmware */
      ptDirectoryInfo->hList = (void*)1;
      strncpy( ptDirectoryInfo->szFilename,
              ptDirListCnf->tData.szName,
              sizeof(ptDirectoryInfo->szFilename));

      ptDirectoryInfo->bFiletype  = ptDirListCnf->tData.bFileType;
      ptDirectoryInfo->ulFilesize = ptDirListCnf->tData.ulFileSize;
    }
    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceFindNextFile, find next file in file system
///   \param hSysdevice         Handle, of a system channel
///   \param ulChannel          Channel number
///   \param ptDirectoryInfo    Directory info
///   \param pfnRecvPktCallback Packet callback
///   \param pvUser             User for packet callback
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xSysdeviceFindNextFile(CIFXHANDLE            hSysdevice,
                                        uint32_t         ulChannel,
                                        CIFX_DIRECTORYENTRY*  ptDirectoryInfo,
                                        PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                        void*                 pvUser)
{
  UNREFERENCED_PARAMETER(pfnRecvPktCallback);
  UNREFERENCED_PARAMETER(pvUser);

  /* NOW, We have a rcX packet and on this time only one connection via one device */
  // NOTE: We ignore the RecvCallback functions pointer, cause we don't receive a async packet

  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  /* Get Device and Channel number from handle */
  if(!InterpretHandle((uint32_t) hSysdevice, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  int32_t lRet = CIFX_NO_ERROR;

  CIFX_PACKET           tcifXPkt      = {0};
  RCX_DIR_LIST_REQ_T*   ptDirListReq  = (RCX_DIR_LIST_REQ_T*)&tcifXPkt;
  uint16_t        usDirNameLen = (uint16_t)(strlen(ptDirectoryInfo->szFilename) + 1);
  uint8_t*        pbRecvData    = NULL;
  uint32_t         ulRecvLen     = 0;


  ptDirListReq->tData.usDirNameLength = usDirNameLen;
  strncpy( (char*)((&ptDirListReq->tData) + 1),
           ptDirectoryInfo->szFilename,
           usDirNameLen);

  ptDirListReq->tHead.ulDest      = RCX_PACKET_DEST_SYSTEM;
  ptDirListReq->tHead.ulSrc       = (uint32_t)hSysdevice;
  ptDirListReq->tHead.ulCmd       = RCX_DIR_LIST_REQ;
  ptDirListReq->tHead.ulLen       = (sizeof(ptDirListReq->tData) + usDirNameLen);
  ptDirListReq->tHead.ulExt       = RCX_PACKET_SEQ_MIDDLE;
  ptDirListReq->tData.ulChannelNo = ulChannel;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                            (BYTE*)ptDirListReq,
                                            sizeof(*ptDirListReq) + ptDirListReq->tData.usDirNameLength,
                                            pbRecvData,
                                            ulRecvLen,
                                            m_pcEndpoint,
                                            RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    RCX_DIR_LIST_CNF_T* ptDirListCnf = reinterpret_cast<RCX_DIR_LIST_CNF_T*>(pbRecvData);

    if (~ptDirListCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
    {
      lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
    } else if( RCX_S_OK == (lRet = ptDirListCnf->tHead.ulSta) )
    {
      if(( ptDirListCnf->tHead.ulExt & RCX_PACKET_SEQ_MASK) == RCX_PACKET_SEQ_LAST)
      {
        /* this is the last packet */
        lRet = CIFX_NO_MORE_ENTRIES;

        /* invalidate handle */
        ptDirectoryInfo->hList = (void*)0;
      } else
      {
        strncpy(ptDirectoryInfo->szFilename,
                ptDirListCnf->tData.szName,
                sizeof(ptDirListCnf->tData.szName));

        ptDirectoryInfo->bFiletype  = ptDirListCnf->tData.bFileType;
        ptDirectoryInfo->ulFilesize = ptDirListCnf->tData.ulFileSize;
      }
    }
    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceDownload download a file to the target
///   \param hSysdevice         Handle, of a system channel
///   \param ulChannel          Channel number
///   \param ulMode             Download mode
///   \param szFileName         File name
///   \param pabFileData        File data buffer
///   \param ulFileSize         File size
///   \param pfnCallback        Progress callback
///   \param pfnRecvPktCallback Receive packet callback
///   \param pvUser             User callback
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xSysdeviceDownload(CIFXHANDLE            hSysdevice,
                                    uint32_t         ulChannel,
                                    uint32_t         ulMode,
                                    char*                 szFileName,
                                    uint8_t*        pabFileData,
                                    uint32_t         ulFileSize,
                                    PFN_PROGRESS_CALLBACK pfnCallback,
                                    PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                    void*                 pvUser)
{
  // NOW, We have a rcX packet and on this time only one connection via one device */
  // NOTE: We ignore the RecvCallback functions pointer, cause we don't receive a async packet

  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  /* Get Device and Channel number from handle */
  if(!InterpretHandle((uint32_t) hSysdevice, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  int32_t            lRet            = CIFX_NO_ERROR;
  uint32_t   ulTransferType  = 0;

  switch(ulMode)
  {
  case DOWNLOAD_MODE_FIRMWARE:
    if( CIFX_NO_ERROR != (lRet = DEV_GetFWTransferTypeFromFileName( szFileName, &ulTransferType)))
      return lRet;

    DEV_RemoveChannelFiles( hSysdevice,
                            ulChannel,
                            this,
                            pfnRecvPktCallback,
                            pvUser,
                            NULL);

    break;

  case DOWNLOAD_MODE_CONFIG:
  case DOWNLOAD_MODE_FILE:
    ulTransferType = RCX_FILE_XFER_FILE;
    break;

  case DOWNLOAD_MODE_LICENSECODE:
    ulTransferType = RCX_FILE_XFER_LICENSE_CODE;
    break;

  default:
    return CIFX_INVALID_PARAMETER;
  }

  lRet = DEV_DownloadFile(hSysdevice,
                          ulChannel,
                          CIFX_MAX_PACKET_SIZE,
                          ulTransferType,
                          szFileName,
                          ulFileSize,
                          pabFileData,
                          this,
                          pfnCallback,
                          pfnRecvPktCallback,
                          pvUser);
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceUpload upload files from target
///   \param hSysdevice         Handle, of a system channel
///   \param ulChannel          Channel number
///   \param ulMode             Download mode
///   \param szFileName         File name
///   \param pabFileData        File data buffer
///   \param pulFileSize        Size of data buffer, returns file size
///   \param pfnCallback        Progress callback
///   \param pfnRecvPktCallback Receive packet callback
///   \param pvUser             User callback
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xSysdeviceUpload(CIFXHANDLE            hSysdevice,
                                  uint32_t         ulChannel,
                                  uint32_t         ulMode,
                                  char*                 szFileName,
                                  uint8_t*        pabFileData,
                                  uint32_t*        pulFileSize,
                                  PFN_PROGRESS_CALLBACK pfnCallback,
                                  PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                  void*                 pvUser)
{
  //  NOW, We have a rcX packet and on this time only one connection via one device */
  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  /* Get Device and Channel number from handle */
  if(!InterpretHandle((uint32_t) hSysdevice, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  int32_t            lRet            = CIFX_NO_ERROR;
  uint32_t   ulTransferType  = 0;


  switch(ulMode)
  {
  case DOWNLOAD_MODE_FIRMWARE:
    if( CIFX_NO_ERROR != (lRet = DEV_GetFWTransferTypeFromFileName( szFileName, &ulTransferType)))
      return lRet;
    break;

  case DOWNLOAD_MODE_CONFIG:
  case DOWNLOAD_MODE_FILE:
    ulTransferType = RCX_FILE_XFER_FILE;
    break;

  default:
    return CIFX_INVALID_PARAMETER;
  }

  lRet =  DEV_UploadFile( hSysdevice,
                          ulChannel,
                          CIFX_MAX_PACKET_SIZE,
                          ulTransferType,
                          szFileName,
                          pulFileSize,
                          pabFileData,
                          this,
                          pfnCallback,
                          pfnRecvPktCallback,
                          pvUser);
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceReset reset the device
///   \param hSysdevice         Handle, of a system channel
///   \param ulTimeout          Function timeout
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xSysdeviceReset ( CIFXHANDLE  hSysdevice, uint32_t ulTimeout)
{
  int32_t lRet = CIFX_NO_ERROR;

  /* Get Device and Channel number from handle */
  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  if(!InterpretHandle((uint32_t) hSysdevice, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  // Check handle is a sysdevice
  if( RCXPACKET_SYSTEMCHANNEL_INTERNAL != bChannel)
    return CIFX_INVALID_CHANNEL;

  // Use Reset to signal the reset event
  lRet = Reset( hSysdevice, CIFX_SYSTEMSTART, ulTimeout);

  /* Reconnect will be handled by calling layer */

  return lRet;
}


/////////////////////////////////////////////////////////////////////////////
/// xChannel functions
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
/// xChannelOpen, open a specific channel
///   \param hDriver    Handle of driver
///   \param szBoard    Board name of the device
///   \param ulChannel  Number of channel, with has to open
///   \param phChannel  Pointer of a handle, for the channel handle
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelOpen(CIFXHANDLE    hDriver,
                              char*         szBoard,
                              uint32_t ulChannel,
                              CIFXHANDLE*   phChannel)
{
  UNREFERENCED_PARAMETER(hDriver);

  /* NOTE: Currently a serial device always has only one "Device" */
  /*       As long as we are running, we should have board. Otherwise */
  /*       no rcXPacket object exists */
  /* NOTE: Channel handling realized by using of m_cvChannel (control structure) */

  int32_t          lRet          = CIFX_NO_ERROR;
  std::string   sBoard(szBoard);
  std::string   sReferenceName(NX_DEFAULT_NAME);
  std::string   sBoardNumber;
  int32_t          lBoardnumber;

  /* Check Board name is valid */
  if( std::string::npos == sBoard.find_first_of( NX_DEFAULT_NAME))
    return CIFX_INVALID_BOARD;

  if(CIFX_NO_ERROR == lRet)
  {
    /* Extract the device number from the board name */
    sBoardNumber = sBoard.substr(sReferenceName.length());
    lBoardnumber = atol(sBoardNumber.c_str());

    /* Extract the device number from class object device name */
    if(std::string::npos == m_szDeviceName.find(NX_DEFAULT_NAME))
      return CIFX_INVALID_BOARD;

    if(CIFX_NO_ERROR == lRet)
    {
      sBoardNumber = m_szDeviceName.substr(sReferenceName.length());

      /* Check board number */
      if( (lBoardnumber != atol(sBoardNumber.c_str())) ||
          (!m_fConnectionValid)                        )
          return CIFX_INVALID_BOARD;


      if(CIFX_NO_ERROR == lRet)
      {
        /*
          HOW TO BUILD A HANDLE

          |--------|--------|--------|--------|
          | lower 16 bit of | Board  | Channel|
          | current this    | number | number |
          | pointer         |        |        |
          |--------|--------|--------|--------|

          NOTE: Sysdevice -> part of channnel number FF
        */

        EnterCriticalSection(&m_tcsChannels);

        CHANNEL_MAP::iterator iter = m_cmChannels.begin();
        while (iter != m_cmChannels.end())
        {
          if (iter->second.ulChannelId == ulChannel)
            break;
          iter++;
        }

        if(iter == m_cmChannels.end())
        {
          lRet = CIFX_INVALID_BOARD;
        } else
        {
          uint8_t bInternalChannelIdx = iter->first;
          *phChannel = (CIFXHANDLE)BuildHandle( (BYTE)lBoardnumber,
                                                bInternalChannelIdx);
          iter->second.ulOpenCount++;
        }

        LeaveCriticalSection(&m_tcsChannels);
        InterlockedIncrement(&m_lRefCount);
      }
    }
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelClose, close a specific channel
///   \param  hChannel  Handle of the specific channel
///   \return CIFX_NO_ERROR  on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelClose( CIFXHANDLE  hChannel)
{
  BYTE bBoard   = 0;
  BYTE bChannel = 0;
  int32_t lRet     = CIFX_NO_ERROR;

  if(!InterpretHandle((uint32_t)hChannel, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  /* Note:  we supporting only one board */
  if(0 != bBoard)
  {
    lRet = CIFX_INVALID_HANDLE;

  } else if(RCXPACKET_SYSTEMCHANNEL_INTERNAL == bChannel)
  {
    lRet = CIFX_INVALID_HANDLE;
  } else
  {
    InterlockedDecrement(&m_lRefCount);

    EnterCriticalSection(&m_tcsChannels);

    CHANNEL_MAP::iterator iter = m_cmChannels.find(bChannel);

    if(iter != m_cmChannels.end())
    {
      CHANNEL_INFO_T& tChannel = iter->second;

      if(0 == tChannel.ulOpenCount)
      {
        lRet = CIFX_DRV_NOT_INITIALIZED;

      } else if(0 == --tChannel.ulOpenCount)
      {
        /* Last one closed the system device, so cleanup queues */

        EnterCriticalSection(&tChannel.tcsRecvLock);

        while(tChannel.cvRecvQueue.size() > 0)
        {
          PHIL_TRANSPORT_PACKET ptPacket = tChannel.cvRecvQueue.front();
          CTransportLayer::FreeTransportFrame(ptPacket);
          tChannel.cvRecvQueue.pop_front();
        }
        LeaveCriticalSection(&tChannel.tcsRecvLock);
      }
    }

    LeaveCriticalSection(&m_tcsChannels);
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelFindFirstFile, find first file in file system
///   \param hChannel           Handle, of a channel
///   \param ptDirectoryInfo    Directory info
///   \param pfnRecvPktCallback Packet callback
///   \param pvUser             User data for callback
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelFindFirstFile( CIFXHANDLE             hChannel,
                                        CIFX_DIRECTORYENTRY*   ptDirectoryInfo,
                                        PFN_RECV_PKT_CALLBACK  pfnRecvPktCallback,
                                        void*                  pvUser)
{
  // Interpret handle
  BYTE     bBoard       = 0;
  BYTE     bChannel     = 0;
  int32_t  lRet         = CIFX_NO_ERROR;
  uint32_t ulChannelId  = 0;

  if( !InterpretHandle((uint32_t)hChannel, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  if(CIFX_NO_ERROR == (lRet = GetChannelId(bChannel, ulChannelId)))
  {
    lRet = xSysdeviceFindFirstFile( hChannel,
                                    ulChannelId,
                                    ptDirectoryInfo,
                                    pfnRecvPktCallback,
                                    pvUser);

  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelFindNextFile, find the next file in file system
///   \param hChannel           Handle, of a channel
///   \param ptDirectoryInfo    Directory info
///   \param pfnRecvPktCallback Packet callback
///   \param pvUser             User data for callback
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelFindNextFile(CIFXHANDLE            hChannel,
                                      CIFX_DIRECTORYENTRY*  ptDirectoryInfo,
                                      PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                      void*                 pvUser)
{
  // Interpret handle
  BYTE     bBoard      = 0;
  BYTE     bChannel    = 0;
  int32_t  lRet        = CIFX_NO_ERROR;
  uint32_t ulChannelId = 0;

  if( !InterpretHandle((uint32_t)hChannel, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  if(CIFX_NO_ERROR == (lRet = GetChannelId(bChannel, ulChannelId)))
  {
    lRet = xSysdeviceFindNextFile(hChannel,
                                  ulChannelId,
                                  ptDirectoryInfo,
                                  pfnRecvPktCallback,
                                  pvUser);

  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelDownload download a file to the target
///   \param hChannel           Handle, of a channel
///   \param ulMode             Download mode
///   \param szFileName         File name
///   \param pabFileData        File data buffer
///   \param ulFileSize         File size
///   \param pfnCallback        Progress callback
///   \param pfnRecvPktCallback Receive packet callback
///   \param pvUser             User callback
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelDownload(CIFXHANDLE            hChannel,
                                  uint32_t         ulMode,
                                  char*                 szFileName,
                                  uint8_t*        pabFileData,
                                  uint32_t         ulFileSize,
                                  PFN_PROGRESS_CALLBACK pfnCallback,
                                  PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                  void*                 pvUser)
{
  // Interpret handle
  BYTE     bBoard      = 0;
  BYTE     bChannel    = 0;
  int32_t  lRet        = CIFX_NO_ERROR;
  uint32_t ulChannelId = 0;

  if( !InterpretHandle((uint32_t)hChannel, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  // Maybe check number of channel
  if(CIFX_NO_ERROR == (lRet = GetChannelId(bChannel, ulChannelId)))
  {
    lRet = xSysdeviceDownload(hChannel,
                              ulChannelId,
                              ulMode,
                              szFileName,
                              pabFileData,
                              ulFileSize,
                              pfnCallback,
                              pfnRecvPktCallback,
                              pvUser);
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelUpload upload a file from the target
///   \param hChannel           Handle, of a channel
///   \param ulMode             Download mode
///   \param szFileName         File name
///   \param pabFileData        File data buffer
///   \param pulFileSize        Size of data buffer, returns file size
///   \param pfnCallback        Progress callback
///   \param pfnRecvPktCallback Receive packet callback
///   \param pvUser             User callback
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelUpload(CIFXHANDLE            hChannel,
                                uint32_t         ulMode,
                                char*                 szFileName,
                                uint8_t*        pabFileData,
                                uint32_t*        pulFileSize,
                                PFN_PROGRESS_CALLBACK pfnCallback,
                                PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                void*                 pvUser)
{
  // Interpret handle
  BYTE     bBoard      = 0;
  BYTE     bChannel    = 0;
  int32_t  lRet        = CIFX_NO_ERROR;
  uint32_t ulChannelId = 0;

  if( !InterpretHandle((uint32_t)hChannel, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  // Maybe check number of channel
  if(CIFX_NO_ERROR == (lRet = GetChannelId(bChannel, ulChannelId)))
  {
    lRet = xSysdeviceUpload(hChannel,
                            ulChannelId,
                            ulMode,
                            szFileName,
                            pabFileData,
                            pulFileSize,
                            pfnCallback,
                            pfnRecvPktCallback,
                            pvUser);
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelGetMBXState, get mail box counter
///   \param  hChannel         Handle, of channel
///   \param  pulRecvPktCount  Pointer of receive packet counter
///   \param  pulSendPktCount  Pointer of sendable packet counter
///   \return CIFX_NO_ERROR,   if success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelGetMBXState(CIFXHANDLE     hChannel,
                                     uint32_t* pulRecvPktCount,
                                     uint32_t* pulSendPktCount)
{
  /* Get Device and Channel number from handle */
  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  if(!InterpretHandle((uint32_t) hChannel, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  if(RCXPACKET_SYSTEMCHANNEL_INTERNAL == bChannel)
    return CIFX_INVALID_HANDLE;

  int32_t lRet = CIFX_NO_ERROR;

  EnterCriticalSection(&m_tcsChannels);
  CHANNEL_MAP::iterator iter = m_cmChannels.find(bChannel);

  if(iter == m_cmChannels.end())
  {
    lRet = CIFX_INVALID_HANDLE;
  } else
  {
    CHANNEL_INFO_T& tChannel = iter->second;

    EnterCriticalSection(&tChannel.tcsRecvLock);

    *pulRecvPktCount = (uint32_t)tChannel.cvRecvQueue.size();
    if(!m_pcTransportLayer->GetSendMbxState( 0, *pulSendPktCount))
      *pulSendPktCount = 0;

    LeaveCriticalSection(&tChannel.tcsRecvLock);

    lRet = CIFX_NO_ERROR;
  }
  LeaveCriticalSection(&m_tcsChannels);

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelPutPacket, send a packet to a channel
///   \param hChannel         Handle, of a channel
///   \param ptSendPkt        Pointer to a packet, which should be send
///   \param ulTimeout        Time out for the send routine
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelPutPacket(CIFXHANDLE     hChannel,
                                   CIFX_PACKET*   ptSendPkt,
                                   uint32_t  ulTimeout)
{
  /* We have a rcX packet and only one connection to one device */
  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  /* Get Device and Channel number from handle */
  if(!InterpretHandle((uint32_t)hChannel, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  if(bChannel == RCXPACKET_SYSTEMCHANNEL_INTERNAL)
    return CIFX_INVALID_HANDLE;

  return xSysdevicePutPacket(hChannel, ptSendPkt, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelGetPacket, get a received  packet from a channel
///   \param hChannel   Handle, of a channel
///   \param ulSize     Size of buffer which stores the received data
///   \param ptRecvPkt  Pointer to a packet, where the received data should be copy
///   \param ulTimeout  Time out for the receive routine
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelGetPacket(CIFXHANDLE     hChannel,
                                   uint32_t  ulSize,
                                   CIFX_PACKET*   ptRecvPkt,
                                   uint32_t  ulTimeout)
{
  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  /* Get Device and Channel number from handle */
  if(!InterpretHandle((uint32_t) hChannel, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  if(bChannel == RCXPACKET_SYSTEMCHANNEL_INTERNAL)
    return CIFX_INVALID_HANDLE;


  return xSysdeviceGetPacket(hChannel, ulSize, ptRecvPkt, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelGetSendPacket read the last send packet back if possible
///   \param hChannel         Handle, of a channel
///   \param ulSize           Size of a buffer
///   \param ptRecvPkt        Packet buffer
///   \return CIFX_NO_ERROR  on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelGetSendPacket( CIFXHANDLE    hChannel,
                                        uint32_t ulSize,
                                        CIFX_PACKET*  ptRecvPkt)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulSize);
  UNREFERENCED_PARAMETER( ptRecvPkt);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelConfigLock lock configuration
///   \param hChannel       Handle, of a channel
///   \param ulCmd          Lock command
///   \param pulState       State buffer
///   \param ulTimeout      Wait timeout
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelConfigLock(  CIFXHANDLE      hChannel,
                                      uint32_t   ulCmd,
                                      uint32_t*  pulState,
                                      uint32_t   ulTimeout)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( pulState);
  UNREFERENCED_PARAMETER( ulTimeout);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelReset reset the target
///   \param hChannel       Handle, of a channel
///   \param ulResetMode    Reset mode
///   \param ulTimeout      Wait timeout
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelReset( CIFXHANDLE    hChannel,
                                uint32_t ulResetMode,
                                uint32_t ulTimeout)
{
  /* Get Device and Channel number from handle */
  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  if(!InterpretHandle((uint32_t) hChannel, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  // Check handle isn't a sysdevice
  if( RCXPACKET_SYSTEMCHANNEL_INTERNAL == bChannel)
    return CIFX_INVALID_CHANNEL;

  int32_t lRet;

  lRet = Reset(hChannel, ulResetMode, ulTimeout);

  /* Reconnect will be handled by calling layer */

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelInfo get information about a specific channel
///   \param hChannel         Handle, of a channel
///   \param ulSize           Size of a buffer
///   \param pvChannelInfo    Pointer to a buffer, where the information should be insert
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelInfo ( CIFXHANDLE    hChannel,
                                uint32_t ulSize,
                                void*         pvChannelInfo)
{
  int32_t  lRet        = CIFX_NO_ERROR;
  uint32_t ulChannelId = 0;
  BYTE     bBoard      = 0;
  BYTE     bChannel    = 0;

  if(ulSize < sizeof(CHANNEL_INFORMATION))
    return CIFX_INVALID_BUFFERSIZE;

  if(!InterpretHandle((uint32_t)hChannel, bBoard, bChannel))
    return CIFX_INVALID_HANDLE;

  if(CIFX_NO_ERROR == (lRet = GetChannelId(bChannel, ulChannelId)))
  {
    lRet = EnumChannel( (uint32_t) bBoard, ulChannelId, ulSize, pvChannelInfo);
    ReadDPMIOInfo( hChannel, (CHANNEL_INFORMATION*)pvChannelInfo);
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelWatchdog handle channel watchdog
///   \param hChannel     Handle, of a channel
///   \param ulCmd        Watchdog command
///   \param pulTrigger   Card trigger information
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelWatchdog(CIFXHANDLE      hChannel,
                                  uint32_t   ulCmd,
                                  uint32_t*  pulTrigger)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( pulTrigger);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelHostState handle host state information
///   \param hChannel     Handle, of a channel
///   \param ulCmd        Watchdog command
///   \param pulState     Actaul state
///   \param pulState     Actaul state
///   \param ulTimeout    Wait timeout
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelHostState( CIFXHANDLE      hChannel,
                                    uint32_t   ulCmd,
                                    uint32_t*  pulState,
                                    uint32_t   ulTimeout)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( pulState);
  UNREFERENCED_PARAMETER( ulTimeout);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelBusState handle device bus state infromation
///   \param hChannel     Handle, of a channel
///   \param ulCmd        Watchdog command
///   \param pulState     Actaul state
///   \param pulState     Actaul state
///   \param ulTimeout    Wait timeout
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelBusState(CIFXHANDLE      hChannel,
                                  uint32_t   ulCmd,
                                  uint32_t*  pulState,
                                  uint32_t   ulTimeout)
{
  /* Get Device and Channel number from handle */
  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  if( !InterpretHandle((uint32_t) hChannel, bBoard, bChannel) ||
      (RCXPACKET_SYSTEMCHANNEL_INTERNAL == bChannel) )
    return CIFX_INVALID_HANDLE;

  int32_t  lRet        = CIFX_NO_ERROR;
  uint32_t ulChannelId = 0;

  if(CIFX_NO_ERROR == (lRet = GetChannelId(bChannel, ulChannelId)))
  {
    switch(ulCmd)
    {
      case CIFX_BUS_STATE_GETSTATE:
      break;

      case CIFX_BUS_STATE_ON:
      case CIFX_BUS_STATE_OFF:
      {
        RCX_START_STOP_COMM_REQ_T  tRcxStartStopCommReq  = {0};
        uint8_t*             pbRecvData            = NULL;
        uint32_t              ulRecvLen             = 0;

        tRcxStartStopCommReq.tHead.ulSrc    = 0;
        tRcxStartStopCommReq.tHead.ulDest   = RCX_PKT_COMM_CHANNEL_TOKEN;
        tRcxStartStopCommReq.tHead.ulCmd    = RCX_START_STOP_COMM_REQ;
        tRcxStartStopCommReq.tHead.ulLen    = sizeof(tRcxStartStopCommReq.tData);
        tRcxStartStopCommReq.tData.ulParam  = (ulCmd == CIFX_BUS_STATE_ON)?1:2;

        lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                                  (BYTE*)&tRcxStartStopCommReq,
                                                  sizeof(tRcxStartStopCommReq),
                                                  pbRecvData,
                                                  ulRecvLen,
                                                  m_pcEndpoint,
                                                  bChannel,
                                                  ulTimeout);

        if(CIFX_NO_ERROR == lRet)
        {
          RCX_START_STOP_COMM_CNF_T* ptCnf = reinterpret_cast<RCX_START_STOP_COMM_CNF_T*>(pbRecvData);

          if (~ptCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
          {
            lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
          } else if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
          {
            *pulState = ulCmd;
          }
          delete [] pbRecvData;
        }
      }
      break;

    default:
      lRet = CIFX_INVALID_COMMAND;
      break;
    }

    /* Read actual BUS state */
    if (CIFX_NO_ERROR == lRet)
    {
      NETX_COMMON_STATUS_BLOCK tCommonStatusBlock = {0};
      if (CIFX_NO_ERROR == ReadCommonStatusBlock(ulChannelId, &tCommonStatusBlock))
      {
        *pulState = (tCommonStatusBlock.ulCommunicationCOS&RCX_COMM_COS_BUS_ON)?CIFX_BUS_STATE_ON:CIFX_BUS_STATE_OFF;
      }
    }
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelIOInfo read channel IO information
///   \param hChannel     Handle, of a channel
///   \param ulCmd        Command
///   \param ulAreaNumber IO area number
///   \param ulSize       Size of the user buffer
///   \param pvData       User buffer to store information
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelIOInfo(CIFXHANDLE    hChannel,
                                uint32_t ulCmd,
                                uint32_t ulAreaNumber,
                                uint32_t ulSize,
                                void*         pvData)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( ulAreaNumber);
  UNREFERENCED_PARAMETER( ulSize);
  UNREFERENCED_PARAMETER( pvData);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelIORead read from the channel input data array
///   \param hChannel     Handle, of a channel
///   \param ulAreaNumber IO area number
///   \param ulOffset     Offset
///   \param ulDataLen    Number of IO bytes to read
///   \param pvData       User buffer to store data
///   \param ulTimeout    Wait timeout
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelIORead(CIFXHANDLE    hChannel,
                                uint32_t ulAreaNumber,
                                uint32_t ulOffset,
                                uint32_t ulDataLen,
                                void*         pvData,
                                uint32_t ulTimeout)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulAreaNumber);
  UNREFERENCED_PARAMETER( ulOffset);
  UNREFERENCED_PARAMETER( ulDataLen);
  UNREFERENCED_PARAMETER( pvData);
  UNREFERENCED_PARAMETER( ulTimeout);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelIOWrite write to channel output data array
///   \param hChannel     Handle, of a channel
///   \param ulAreaNumber IO area number
///   \param ulOffset     Offset
///   \param ulDataLen    Number of IO bytes to write
///   \param pvData       User data to be written
///   \param ulTimeout    Wait timeout
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelIOWrite(CIFXHANDLE     hChannel,
                                 uint32_t  ulAreaNumber,
                                 uint32_t  ulOffset,
                                 uint32_t  ulDataLen,
                                 void*          pvData,
                                 uint32_t  ulTimeout)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulAreaNumber);
  UNREFERENCED_PARAMETER( ulOffset);
  UNREFERENCED_PARAMETER( ulDataLen);
  UNREFERENCED_PARAMETER( pvData);
  UNREFERENCED_PARAMETER( ulTimeout);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelIOReadSendData read from the channel output data array
///   \param hChannel     Handle, of a channel
///   \param ulAreaNumber IO area number
///   \param ulOffset     Offset
///   \param ulDataLen    Number of IO bytes to read
///   \param pvData       User buffer to store data
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelIOReadSendData(CIFXHANDLE    hChannel,
                                        uint32_t ulAreaNumber,
                                        uint32_t ulOffset,
                                        uint32_t ulDataLen,
                                        void*         pvData)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulAreaNumber);
  UNREFERENCED_PARAMETER( ulOffset);
  UNREFERENCED_PARAMETER( ulDataLen);
  UNREFERENCED_PARAMETER( pvData);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelControlBlock read/write the channel control area
///   \param hChannel     Handle, of a channel
///   \param ulCmd        Command
///   \param ulOffset     Offset in the control block
///   \param ulDataLen    Number of IO bytes to read
///   \param pvData       User buffer to store data
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelControlBlock(CIFXHANDLE    hChannel,
                                      uint32_t ulCmd,
                                      uint32_t ulOffset,
                                      uint32_t ulDataLen,
                                      void*         pvData)
{
  /* Get Device and Channel number from handle */
  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  if( !InterpretHandle((uint32_t) hChannel, bBoard, bChannel) ||
      (RCXPACKET_SYSTEMCHANNEL_INTERNAL == bChannel) )
    return CIFX_INVALID_HANDLE;

  int32_t  lRet        = CIFX_NO_ERROR;
  uint32_t ulChannelId = 0;

  if(CIFX_NO_ERROR == (lRet = GetChannelId(bChannel, ulChannelId)))
  {
    if (CIFX_CMD_READ_DATA != ulCmd)
    {
      lRet = CIFX_INVALID_COMMAND;
    } else
    {
      NETX_CONTROL_BLOCK tControl;
      if (CIFX_NO_ERROR == (lRet = ReadChannelControlBlock(ulChannelId, &tControl)))
      {
        if(sizeof(tControl) >= (ulOffset + ulDataLen))
        {
          /* Copy data to user buffer */
          memcpy(pvData, &tControl + ulOffset, ulDataLen);
        }else
        {
          lRet = CIFX_INVALID_ACCESS_SIZE;
        }
      }
    }
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelCommonStatusBlock read from the channel common status area
///   \param hChannel     Handle, of a channel
///   \param ulCmd        Command
///   \param ulOffset     Offset in the status area
///   \param ulDataLen    Number of IO bytes to read
///   \param pvData       User buffer to store data
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelCommonStatusBlock( CIFXHANDLE    hChannel,
                                            uint32_t ulCmd,
                                            uint32_t ulOffset,
                                            uint32_t ulDataLen,
                                            void*         pvData)
{
  /* Get Device and Channel number from handle */
  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  if( !InterpretHandle((uint32_t) hChannel, bBoard, bChannel) ||
      (RCXPACKET_SYSTEMCHANNEL_INTERNAL == bChannel) )
    return CIFX_INVALID_HANDLE;

  int32_t  lRet        = CIFX_NO_ERROR;
  uint32_t ulChannelId = 0;

  if(CIFX_NO_ERROR == (lRet = GetChannelId(bChannel, ulChannelId)))
  {
    if (CIFX_CMD_READ_DATA != ulCmd)
    {
      lRet = CIFX_INVALID_COMMAND;
    } else
    {
      NETX_COMMON_STATUS_BLOCK tStatus = {0};
      if (CIFX_NO_ERROR == (lRet = ReadCommonStatusBlock(ulChannelId, &tStatus)))
      {
        if(sizeof(tStatus) >= (ulOffset + ulDataLen))
        {
          /* Copy data to user buffer */
          memcpy(pvData, &tStatus + ulOffset, ulDataLen);
        }else
        {
          lRet = CIFX_INVALID_ACCESS_SIZE;
        }
      }
    }
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelExtendedStatusBlock read from the channel extended status area
///   \param hChannel     Handle, of a channel
///   \param ulCmd        Command
///   \param ulOffset     Offset in the status area
///   \param ulDataLen    Number of IO bytes to read
///   \param pvData       User buffer to store data
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelExtendedStatusBlock( CIFXHANDLE    hChannel,
                                              uint32_t ulCmd,
                                              uint32_t ulOffset,
                                              uint32_t ulDataLen,
                                              void*         pvData)
{
  /* Get Device and Channel number from handle */
  BYTE bBoard   = 0;
  BYTE bChannel = 0;

  if( !InterpretHandle((uint32_t) hChannel, bBoard, bChannel) ||
      (RCXPACKET_SYSTEMCHANNEL_INTERNAL == bChannel) )
    return CIFX_INVALID_HANDLE;

  int32_t  lRet        = CIFX_NO_ERROR;
  uint32_t ulChannelId = 0;

  if(CIFX_NO_ERROR == (lRet = GetChannelId(bChannel, ulChannelId)))
  {
    switch(ulCmd)
    {
      case CIFX_CMD_READ_DATA :
      {
        RCX_DPM_GET_EXTENDED_STATE_REQ_T tRcxExtStatusReq      = {0};
        uint8_t*                    pbRecvData            = NULL;
        uint32_t                     ulRecvLen             = 0;

        tRcxExtStatusReq.tHead.ulSrc    = 0;
        tRcxExtStatusReq.tHead.ulDest   = RCX_PACKET_DEST_SYSTEM;
        tRcxExtStatusReq.tHead.ulCmd    = RCX_DPM_GET_EXTENDED_STATE_REQ;
        tRcxExtStatusReq.tHead.ulLen    = sizeof(tRcxExtStatusReq.tData.ulOffset) +
                                          sizeof(tRcxExtStatusReq.tData.ulDataLen) +
                                          sizeof(tRcxExtStatusReq.tData.ulChannelIndex);

        tRcxExtStatusReq.tData.ulOffset       = ulOffset;
        tRcxExtStatusReq.tData.ulDataLen      = ulDataLen;
        tRcxExtStatusReq.tData.ulChannelIndex = ulChannelId;

        lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_RCX_PACKET,
                                                  (BYTE*)&tRcxExtStatusReq,
                                                  sizeof(tRcxExtStatusReq),
                                                  pbRecvData,
                                                  ulRecvLen,
                                                  m_pcEndpoint,
                                                  RCXPACKET_SYSTEMCHANNEL_INTERNAL,
                                                  CIFX_TO_CONT_PACKET);

        if(CIFX_NO_ERROR == lRet)
        {
          RCX_DPM_GET_EXTENDED_STATE_CNF_T* ptCnf = reinterpret_cast<RCX_DPM_GET_EXTENDED_STATE_CNF_T*>(pbRecvData);

          if (~ptCnf->tHead.ulCmd&RCX_MSK_PACKET_ANSWER)
          {
            lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
          } else if(CIFX_NO_ERROR == (lRet = ptCnf->tHead.ulSta))
          {
            if(ptCnf->tData.ulDataLen >= (ulOffset + ulDataLen))
            {
              memcpy(pvData, &ptCnf->tData.abData, ulDataLen);
            } else
            {
              lRet = CIFX_INVALID_ACCESS_SIZE;
            }
          }
          delete [] pbRecvData;
        }
      }
      break;

    default:
      lRet = CIFX_INVALID_COMMAND;
      break;
    }
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelUserBlock read/write the channel user area if available
///   \param hChannel     Handle, of a channel
///   \param ulAreaNumber Number of the user data area
///   \param ulCmd        Command
///   \param ulOffset     Offset in the area
///   \param ulDataLen    Number of IO bytes to read/write
///   \param pvData       User buffer to store data
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CrcXPacket::xChannelUserBlock(CIFXHANDLE     hChannel,
                                   uint32_t  ulAreaNumber,
                                   uint32_t  ulCmd,
                                   uint32_t  ulOffset,
                                   uint32_t  ulDataLen,
                                   void*          pvData)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulAreaNumber);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( ulOffset);
  UNREFERENCED_PARAMETER( ulDataLen);
  UNREFERENCED_PARAMETER( pvData);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/////////////////////////////////////////////////////////////////////////////
/// Populates a board info structure from rcX internal data
///   \param ptBoardInfo Handle, of a channel
/////////////////////////////////////////////////////////////////////////////
void CrcXPacket::GenerateBoardInformation(BOARD_INFORMATION* ptBoardInfo)
{
  ptBoardInfo->lBoardError          = 0;
  ptBoardInfo->ulBoardID            = 0;
  ptBoardInfo->ulSystemError        = 0;
  ptBoardInfo->ulPhysicalAddress    = 0;
  ptBoardInfo->bIrqEnabled          = 0;
  ptBoardInfo->ulIrqNumber          = 0;

  EnterCriticalSection(&m_tcsChannels);
  ptBoardInfo->ulChannelCnt         = (uint32_t)(m_cmChannels.size() - 1);
  LeaveCriticalSection(&m_tcsChannels);

  ptBoardInfo->ulDpmTotalSize       = m_tSystemInfo.ulDpmTotalSize;

  strncpy ( ptBoardInfo->abBoardName,  m_szDeviceName.c_str(), sizeof(ptBoardInfo->abBoardName));
  memset  ( ptBoardInfo->abBoardAlias, 0, sizeof(ptBoardInfo->abBoardAlias));
  memcpy  ( &ptBoardInfo->tSystemInfo, &m_tSystemInfo,       sizeof(m_tSystemInfo));
}

/////////////////////////////////////////////////////////////////////////////
/// Populates a channel info structure from rcX internal channel info data
///   \param tChannelInfo  rcX internal channel data
///   \param ptChannelInfo Channel info structure
/////////////////////////////////////////////////////////////////////////////
void CrcXPacket::GenerateChannelInfo(CHANNEL_INFO_T& tChannelInfo, CHANNEL_INFORMATION* ptChannelInfo)
{
  // Get device name
  strncpy ( ptChannelInfo->abBoardName,
            m_szDeviceName.c_str(),
            (uint32_t) sizeof(ptChannelInfo->abBoardName));  /* Global board name              */

  ptChannelInfo->ulDeviceNumber = m_tSystemInfo.ulDeviceNumber;   /* Global board device number     */
  ptChannelInfo->ulSerialNumber = m_tSystemInfo.ulSerialNumber;   /* Global board serial number     */

  ptChannelInfo->usFWMajor      = tChannelInfo.tFirmwareInfo.tFwVersion.usMajor;     /* Major Version of Channel Firmware  */
  ptChannelInfo->usFWMinor      = tChannelInfo.tFirmwareInfo.tFwVersion.usMinor;     /* Minor Version of Channel Firmware  */
  ptChannelInfo->usFWBuild      = tChannelInfo.tFirmwareInfo.tFwVersion.usBuild;     /* Build number of Channel Firmware   */
  ptChannelInfo->usFWRevision   = tChannelInfo.tFirmwareInfo.tFwVersion.usRevision;  /* Revision of Channel Firmware       */
  ptChannelInfo->bFWNameLength  = tChannelInfo.tFirmwareInfo.tFwName.bNameLength;    /* Length  of FW Name                 */

  memcpy   (  ptChannelInfo->abFWName,                                                                       /* Firmware Name                      */
              tChannelInfo.tFirmwareInfo.tFwName.abName,
              sizeof(ptChannelInfo->abFWName));

  ptChannelInfo->usFWYear         = tChannelInfo.tFirmwareInfo.tFwDate.usYear;       /* Build Year of Firmware             */
  ptChannelInfo->bFWMonth         = tChannelInfo.tFirmwareInfo.tFwDate.bMonth;       /* Build Month of Firmware (1..12)    */
  ptChannelInfo->bFWDay           = tChannelInfo.tFirmwareInfo.tFwDate.bDay;         /* Build Day of Firmware (1..31)      */
  ptChannelInfo->ulChannelError   = 0;                                                                       /* Channel error                      */
  ptChannelInfo->ulOpenCnt        = tChannelInfo.ulOpenCount;
  ptChannelInfo->ulMailboxSize    = CIFX_MAX_PACKET_SIZE;


  /* Read communication COS flags */
  NETX_COMMON_STATUS_BLOCK tStatus = {0};
  if (CIFX_NO_ERROR == ReadCommonStatusBlock(tChannelInfo.ulChannelId, &tStatus))
  {
    ptChannelInfo->ulDeviceCOSFlags = tStatus.ulCommunicationCOS;
    ptChannelInfo->ulChannelError   = tStatus.ulCommunicationError;
    tChannelInfo.ulPDInHskMode      = tStatus.bPDInHskMode;
    tChannelInfo.ulPDOutHskMode     = tStatus.bPDOutHskMode;
  }

  /* Read applicaton COS flags */
  NETX_CONTROL_BLOCK tControl = {0};
  if (CIFX_NO_ERROR == ReadChannelControlBlock(tChannelInfo.ulChannelId, &tControl))
  {
    ptChannelInfo->ulHostCOSFlags = tControl.ulApplicationCOS;
  }

  /* Read actual netX and host communication flags */
  ReadComFlags(tChannelInfo.ulAreaNr, &ptChannelInfo->ulNetxFlags, &ptChannelInfo->ulHostFlags);

  ptChannelInfo->ulHskSize      = tChannelInfo.ulHSKFlagSize;
  ptChannelInfo->ulPutPacketCnt = tChannelInfo.ulSendPacketCnt;
  ptChannelInfo->ulGetPacketCnt = tChannelInfo.ulRecvPacketCnt;
}
